// popup.js - Gestion authentification de l'extension
// SERVER_URL est défini dans config.js

document.addEventListener('DOMContentLoaded', async () => {
  // Charger la config distante avant toute action
  await loadRemoteConfig();

  const loginView = document.getElementById('loginView');
  const settingsView = document.getElementById('settingsView');

  // Site button
  document.getElementById('siteBtn').addEventListener('click', () => {
    chrome.tabs.create({ url: SERVER_URL });
  });

  // Login elements
  const loginEmail = document.getElementById('loginEmail');
  const loginPassword = document.getElementById('loginPassword');
  const loginBtn = document.getElementById('loginBtn');
  const loginError = document.getElementById('loginError');

  // Settings elements
  const usernameDisplay = document.getElementById('usernameDisplay');
  const logoutBtn = document.getElementById('logoutBtn');
  const serverSelect = document.getElementById('serverSelect');
  const updateBtn = document.getElementById('updateBtn');
  const versionInfo = document.getElementById('versionInfo');

  // Load stored data and decide which view to show
  chrome.storage.sync.get(
    { apiToken: '', username: '', gameServer: 'Totemia' },
    (data) => {
      if (data.apiToken) {
        showSettings(data.username);
      } else {
        showLogin();
      }
      serverSelect.value = data.gameServer || 'Totemia';
    },
  );

  // Server change handler
  serverSelect.addEventListener('change', () => {
    chrome.storage.sync.set({ gameServer: serverSelect.value });
  });

  // Check for extension updates
  checkForUpdate();

  // Update button handler
  updateBtn.addEventListener('click', () => {
    if (!updateBtn.disabled && updateBtn.dataset.updateUrl) {
      chrome.tabs.create({ url: updateBtn.dataset.updateUrl });
    }
  });

  // Login handler
  loginBtn.addEventListener('click', async () => {
    const email = loginEmail.value.trim();
    const password = loginPassword.value.trim();

    if (!email || !password) {
      showError('Veuillez remplir tous les champs.');
      return;
    }

    loginBtn.disabled = true;
    loginBtn.textContent = 'Connexion...';
    hideError();

    try {
      const response = await fetch(`${SERVER_URL}/api/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, password }),
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || `Erreur ${response.status}`);
      }

      // Store token and username
      chrome.storage.sync.set(
        {
          apiToken: data.token,
          username: data.user.username,
        },
        () => {
          showSettings(data.user.username);
        },
      );
    } catch (err) {
      showError(err.message);
    } finally {
      loginBtn.disabled = false;
      loginBtn.textContent = 'Se connecter';
    }
  });

  // Logout handler
  logoutBtn.addEventListener('click', () => {
    chrome.storage.sync.remove(['apiToken', 'username'], () => {
      loginPassword.value = '';
      showLogin();
    });
  });

  function showLogin() {
    loginView.classList.remove('hidden');
    settingsView.classList.add('hidden');
  }

  function showSettings(username) {
    loginView.classList.add('hidden');
    settingsView.classList.remove('hidden');
    usernameDisplay.textContent = username || '';
  }

  function showError(msg) {
    loginError.textContent = msg;
    loginError.style.display = 'block';
  }

  function hideError() {
    loginError.style.display = 'none';
  }

  async function checkForUpdate() {
    const currentVersion = chrome.runtime.getManifest().version;
    versionInfo.textContent = `v${currentVersion}`;

    try {
      const res = await fetch(`${SERVER_URL}/api/extension/version`, { cache: 'no-cache' });
      if (!res.ok) return;
      const data = await res.json();
      const latestVersion = data.version;

      if (latestVersion && compareVersions(latestVersion, currentVersion) > 0) {
        updateBtn.disabled = false;
        updateBtn.textContent = `Mettre a jour (v${latestVersion})`;
        if (data.downloadUrl) {
          updateBtn.dataset.updateUrl = data.downloadUrl;
        } else {
          updateBtn.dataset.updateUrl = SERVER_URL;
        }
        versionInfo.textContent = `v${currentVersion} → v${latestVersion} disponible`;
        versionInfo.style.color = '#16a34a';
      } else {
        versionInfo.textContent = `v${currentVersion} — a jour`;
      }
    } catch {
      // Silently fail — button stays disabled
    }
  }

  function compareVersions(a, b) {
    const pa = a.split('.').map(Number);
    const pb = b.split('.').map(Number);
    for (let i = 0; i < Math.max(pa.length, pb.length); i++) {
      const na = pa[i] || 0;
      const nb = pb[i] || 0;
      if (na > nb) return 1;
      if (na < nb) return -1;
    }
    return 0;
  }
});
