// content.js - Multi-scan concurrent avec notifications

(function () {
  'use strict';

  let container = null;
  let pendingNotif = null;
  let pendingCount = 0;
  const pendingSessions = new Set();

  function isExtensionValid() {
    try {
      return !!chrome.runtime?.id;
    } catch {
      return false;
    }
  }

  chrome.runtime.onMessage.addListener((message) => {
    if (message.action !== 'scanUpdate') return;
    if (!pendingSessions.has(message.sessionId)) return;
    pendingSessions.delete(message.sessionId);
    pendingCount = Math.max(0, pendingCount - 1);

    updatePendingNotif();
  });

  function getContainer() {
    if (container && document.body.contains(container)) return container;
    container = document.createElement('div');
    container.id = 'scanachat-container';
    Object.assign(container.style, {
      position: 'fixed',
      top: '16px',
      right: '16px',
      zIndex: '2147483647',
      display: 'flex',
      flexDirection: 'column',
      alignItems: 'flex-end',
      gap: '8px',
      fontFamily: 'system-ui, -apple-system, sans-serif',
    });
    document.body.appendChild(container);
    return container;
  }

  function makeEl(text, bg) {
    const el = document.createElement('div');
    Object.assign(el.style, {
      background: bg,
      color: '#f1f5f9',
      padding: '12px 16px',
      borderRadius: '10px',
      fontSize: '13px',
      fontWeight: '500',
      boxShadow: '0 4px 16px rgba(0,0,0,0.4)',
      width: 'max-content',
      maxWidth: '300px',
      transition: 'opacity 0.3s, transform 0.3s',
      transform: 'translateX(100%)',
      opacity: '0',
    });
    el.innerHTML = `<span class="scanachat-label">${text}</span>`;
    getContainer().appendChild(el);
    requestAnimationFrame(() => {
      el.style.transform = 'translateX(0)';
      el.style.opacity = '1';
    });
    return el;
  }

  function dismiss(el) {
    el.style.opacity = '0';
    el.style.transform = 'translateX(100%)';
    setTimeout(() => el.remove(), 300);
  }

  function updatePendingNotif() {
    if (pendingCount <= 0) {
      if (pendingNotif) { dismiss(pendingNotif); pendingNotif = null; }
      return;
    }
    const text = pendingCount === 1
      ? 'Analyse de la boutique...'
      : `Analyse de ${pendingCount} boutiques...`;
    if (!pendingNotif || !document.body.contains(pendingNotif)) {
      pendingNotif = makeEl(text, '#1e293b');
    } else {
      const label = pendingNotif.querySelector('.scanachat-label');
      if (label) label.textContent = text;
    }
  }

  document.addEventListener('keydown', (event) => {
    if (event.key !== 'Tab') return;
    const tag = event.target.tagName.toLowerCase();
    if (tag === 'input' || tag === 'textarea' || tag === 'select' || event.target.isContentEditable) return;

    event.preventDefault();
    event.stopPropagation();
    if (!isExtensionValid()) return;

    pendingCount++;
    updatePendingNotif();

    try {
      chrome.runtime.sendMessage(
        { action: 'captureAndScan', devicePixelRatio: window.devicePixelRatio || 1 },
        (response) => {
          if (chrome.runtime.lastError) {
            pendingCount = Math.max(0, pendingCount - 1);
            updatePendingNotif();
            return;
          }
          if (response && response.error) {
            pendingCount = Math.max(0, pendingCount - 1);
            updatePendingNotif();
          } else if (response && response.success) {
            pendingSessions.add(response.sessionId);
          }
        }
      );
    } catch {
      pendingCount = Math.max(0, pendingCount - 1);
      updatePendingNotif();
    }
  }, true);
})();
