// background.js - Service worker : capture, envoie au backend, poll le statut
// SERVER_URL est défini dans config.js
importScripts('config.js');

const CROP_WIDTH = 434;
const CROP_HEIGHT = 455;

// Charger la config distante au démarrage du service worker
loadRemoteConfig();

// Active scans: Map<sessionId, { statusUrl, resultsPageUrl, tabId }>
const activeScans = new Map();
let pollTimer = null;

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'captureAndScan') {
    handleCaptureAndScan(sender.tab.id, message.devicePixelRatio || 1)
      .then((result) => {
        if (result.success) {
          activeScans.set(result.sessionId, {
            statusUrl: result.statusUrl,
            resultsPageUrl: result.resultsPageUrl,
            tabId: sender.tab.id,
          });
          startPolling();
        }
        sendResponse(result);
      })
      .catch((err) => sendResponse({ error: err.message }));
    return true;
  }

  if (message.action === 'getActiveScans') {
    sendResponse({ scans: Object.fromEntries(activeScans) });
    return false;
  }

  return false;
});

function startPolling() {
  if (pollTimer !== null) return;
  pollTimer = setInterval(batchPoll, 3000);
}

function stopPolling() {
  if (pollTimer !== null) {
    clearInterval(pollTimer);
    pollTimer = null;
  }
}

async function batchPoll() {
  if (activeScans.size === 0) {
    stopPolling();
    return;
  }

  const ids = Array.from(activeScans.keys()).join(',');

  try {
    const settings = await chrome.storage.sync.get({ apiToken: '' });
    const headers = {};
    if (settings.apiToken) {
      headers['Authorization'] = `Bearer ${settings.apiToken}`;
    }

    const r = await fetch(`${SERVER_URL}/api/scan/batch-status?ids=${ids}`, { headers });
    if (r.status === 502 || r.status === 503) return; // retry next tick

    const data = await r.json();

    for (const [sessionId, scan] of activeScans) {
      const status = data[sessionId];
      if (!status) continue;

      if (status.status === 'completed' || status.status === 'error') {
        // Notify the content script
        try {
          chrome.tabs.sendMessage(scan.tabId, {
            action: 'scanUpdate',
            sessionId,
            ...status,
            resultsPageUrl: scan.resultsPageUrl,
          });
        } catch { /* tab may be closed */ }
        activeScans.delete(sessionId);
      }
    }

    if (activeScans.size === 0) stopPolling();
  } catch {
    // Network error, retry next tick
  }
}

async function handleCaptureAndScan(tabId, dpr) {
  const settings = await chrome.storage.sync.get({ apiToken: '', gameServer: 'Totemia' });
  const serverUrl = SERVER_URL;
  const gameServer = settings.gameServer || 'Totemia';

  const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });

  const cropW = Math.round(CROP_WIDTH * dpr);
  const cropH = Math.round(CROP_HEIGHT * dpr);
  const croppedBase64 = await cropImage(dataUrl, cropW, cropH);

  const headers = { 'Content-Type': 'application/json' };
  if (settings.apiToken) {
    headers['Authorization'] = `Bearer ${settings.apiToken}`;
  }

  const response = await fetch(`${serverUrl}/api/scan`, {
    method: 'POST',
    headers,
    body: JSON.stringify({ image: croppedBase64, server: gameServer }),
  });

  if (!response.ok) {
    const errData = await response.json().catch(() => ({}));
    throw new Error(errData.error || `Serveur a retourne ${response.status}`);
  }

  const result = await response.json();

  return {
    success: true,
    sessionId: result.sessionId,
    statusUrl: `${serverUrl}/api/scan/${result.sessionId}/status`,
    resultsPageUrl: `${serverUrl}/scan/${result.sessionId}`,
  };
}

async function cropImage(dataUrl, targetWidth, targetHeight) {
  const response = await fetch(dataUrl);
  const blob = await response.blob();
  const bitmap = await createImageBitmap(blob);

  const sourceWidth = bitmap.width;
  const sourceHeight = bitmap.height;

  const cropX = Math.max(0, Math.floor((sourceWidth - targetWidth) / 2));
  const cropY = Math.max(0, Math.floor((sourceHeight - targetHeight) / 2));
  const cropW = Math.min(targetWidth, sourceWidth);
  const cropH = Math.min(targetHeight, sourceHeight);

  const canvas = new OffscreenCanvas(cropW, cropH);
  const ctx = canvas.getContext('2d');
  ctx.drawImage(bitmap, cropX, cropY, cropW, cropH, 0, 0, cropW, cropH);
  bitmap.close();

  const pngBlob = await canvas.convertToBlob({ type: 'image/png' });
  const arrayBuffer = await pngBlob.arrayBuffer();
  const bytes = new Uint8Array(arrayBuffer);
  let binary = '';
  for (let i = 0; i < bytes.length; i++) {
    binary += String.fromCharCode(bytes[i]);
  }

  return 'data:image/png;base64,' + btoa(binary);
}
